
vec3 MoonFlux = vec3(abs(moonPhase - 4.0) * 0.25 + 0.2) * (NIGHT_BRIGHTNESS + nightVision * 0.02);

const float planetRadius = 6371e3;

#ifdef AURORA
	float auroraAmount = smoothstep(0.0, 0.2, -worldSunVector.y) * AURORA_STRENGTH;
#endif

vec4 ToSH(float value, vec3 dir) {
	const vec2 foo = vec2(0.5 * PI * sqrt(rPI), 0.3849 * PI * sqrt(0.75 * rPI));
    return vec4(foo.x, foo.y * dir.yzx) * value;
}

vec3 FromSH(vec4 cR, vec4 cG, vec4 cB, vec3 lightDir) {
	const vec2 foo = vec2(0.5 * sqrt(rPI), sqrt(0.75 * rPI));
    vec4 sh = vec4(foo.x, foo.y * lightDir.yzx);

    return vec3(dot(sh, cR), dot(sh, cG), dot(sh, cB));
}

//#define HORIZON_THING

//const vec3 Rayleigh = vec3(0.21, 0.45, 0.99);
const vec3 Rayleigh = vec3(0.17, 0.40, 0.99);

float AtmosphereMie = 0.005 + wetness * 0.2;
const float AtmosphereDensity = 0.07;
const float AtmosphereDensityFalloff = 0.3;
const float AtmosphereExtent = 80.0;

vec3 MplusR = AtmosphereMie + Rayleigh;

// polynomial smooth min (k = 0.1); https://www.iquilezles.org/www/articles/smin/smin.htm
float SmoothMin(in float a, in float b, in float k) {
    float h = saturate(0.5 + 0.5 * (b - a) / k);
    return mix(b, a, h) - k * h * oneMinus(h);
}

float SmoothMax(in float a, in float b, in float k) {
    return -SmoothMin(-a, -b, k);
}

float HenyeyGreensteinPhase(in float cosTheta, in const float g) {
	const float gg = g * g;
    float phase = 1.0 + gg - 2.0 * g * cosTheta;
    return oneMinus(gg) / (4.0 * PI * phase * sqrt(phase));
}

float CornetteShanksPhase(in float cosTheta, in const float g) {
	const float gg = g * g;
  	float a = oneMinus(gg) * rcp(2.0 + gg) * 3.0 * rPI;
  	float b = (1.0 + sqr(cosTheta)) * pow((1.0 + gg - 2.0 * g * cosTheta), -1.5);
  	return a * b * 0.125;
}

float MiePhaseClouds(in float cosTheta, in const vec3 g, in const vec3 w) {
	const vec3 gg = g * g;
	vec3 a = (0.75 * oneMinus(gg)) * rcp(2.0 + gg)/* * rTAU*/;
	vec3 b = (1.0 + sqr(cosTheta)) * pow(1.0 + gg - 2.0 * g * cosTheta, vec3(-1.5));

	return dot(a * b, w) / (w.x + w.y + w.z);
}

vec3 DoNightEye(in vec3 color) {
	float luminance = GetLuminance(color);
	float rodFactor = exp2(-luminance * 1e2);
	return mix(color, luminance * vec3(0.72, 0.95, 1.2), rodFactor);
}
/*
// https://advances.realtimerendering.com/s2021/jpatry_advances2021/index.html
// https://citeseerx.ist.psu.edu/viewdoc/download?doi=10.1.1.40.9608&rep=rep1&type=pdf
vec4 rgb_to_lmsr(vec3 c) {
    const mat4x3 m = mat4x3(
        0.31670331, 0.70299344, 0.08120592, 
        0.10129085, 0.72118661, 0.12041039, 
        0.01451538, 0.05643031, 0.53416779, 
        0.01724063, 0.60147464, 0.40056206);
    return c * m;
}
vec3 lms_to_rgb(vec3 c) {
    const mat3 m = mat3(
         4.57829597, -4.48749114,  0.31554848, 
        -0.63342362,  2.03236026, -0.36183302, 
        -0.05749394, -0.09275939,  1.90172089);
    return c * m;
}

// https://www.ncbi.nlm.nih.gov/pmc/articles/PMC2630540/pdf/nihms80286.pdf
vec3 apply_purkinje_shift(vec3 c) {
    const vec3 m = vec3(0.63721, 0.39242, 1.6064);
    const float K = 45.0;
    const float S = 10.0;
    const float k3 = 0.6;
    const float k5 = 0.2;
    const float k6 = 0.29;
    const float rw = 0.139;
    const float p = 0.6189;

    const mat4x3 rgb_to_lmsr = mat4x3(
        0.31670331, 0.70299344, 0.08120592, 
        0.10129085, 0.72118661, 0.12041039, 
        0.01451538, 0.05643031, 0.53416779, 
        0.01724063, 0.60147464, 0.40056206);
    const mat3 lms_to_rgb = mat3(
         4.57829597, -4.48749114,  0.31554848, 
        -0.63342362,  2.03236026, -0.36183302, 
        -0.05749394, -0.09275939,  1.90172089);

    //vec4 lmsr = rgb_to_lmsr(c);
    vec4 lmsr = c * rgb_to_lmsr;

    vec3 g = vec3(1.0) / sqrt(vec3(1.0) + (vec3(0.33) / m) * (lmsr.xyz + vec3(k5, k5, k6) * lmsr.w));
    
    float rc_gr = (K / S) * ((1.0 + rw * k3) * g.y / m.y - (k3 + rw) * g.x / m.x) * k5 * lmsr.w;
    float rc_by = (K / S) * (k6 * g.z / m.z - k3 * (p * k5 * g.x / m.x + (1.0 - p) * k5 * g.y / m.y)) * lmsr.w;
    float rc_lm = K * (p * g.x / m.x + (1.0 - p) * g.y / m.y) * k5 * lmsr.w;

    vec3 lms_gain = vec3(-0.5 * rc_gr + 0.5 * rc_lm, 0.5 * rc_gr + 0.5 * rc_lm, rc_by + rc_lm);
    vec3 rgb_gain = lms_gain * lms_to_rgb;

    return c + rgb_gain;
}
*/
/*
float TotalAtmosphereDensity(float dirY, const bool considerLand)
{
	if (!considerLand || (considerLand && dirY > 0.0))
	{
		return 1.0 / (max(0.0, dirY) * 6.0 + 0.03);
	}
	else
	{
		return 1.0 / (max(0.0, -dirY) * 50.0 + 0.4);
	}
}
*/
vec3 AtmosphereAbsorption(in vec3 dir, in float depth) {
	if (dir.y < 0.0 && depth > 10.0) return vec3(0.0);

	float rDFv = AtmosphereDensityFalloff * dir.y;

	vec3 absorption = expf((-MplusR/* - wetness * 0.5*/) * (AtmosphereDensity / rDFv) * oneMinus(expf(-rDFv * depth)));

	return absorption;
}

vec3 SunlightColorFromSunVector(in vec3 worldSunVector) {
	vec3 color = AtmosphereAbsorption(worldSunVector, AtmosphereExtent);
		 color *= saturate(worldSunVector.y * 40.0);
	return color * exp2(-Rayleigh * 0.2);
}

vec3 Atmosphere(in vec3 worldViewVector, in vec3 worldSunVector, in float mieAmount, in float depthFactor) {
	float LdotV = dot(worldViewVector, worldSunVector);
	//float LdotV2 = LdotV * LdotV;
	float LdotV_2 = 0.5 - LdotV * 0.5;

	float x = depthFactor;
	float s = SmoothMax(0.01, worldSunVector.y, 0.07);
	float v = pow(abs(worldViewVector.y), 1.0 + LdotV_2 * 5e-4 / (s + 5e-4)) * sign(worldViewVector.y);
	float energyFade = expf(min(0.0, worldSunVector.y) * 1e2);

	float M = AtmosphereMie;
	float rDF = AtmosphereDensityFalloff - LdotV_2 * max0(-worldSunVector.y + 0.01) * 4.0;

	//#ifdef HORIZON_THING
	#ifdef HORIZON_THING
		float floorDist = min(x, 0.5 / (-worldViewVector.y + 0.00918));
		if (worldViewVector.y < 1e-6) x = mix(floorDist, x, wetness);
	#endif
		M += 0.007 / (saturate(worldViewVector.y) + 0.05);
	//#endif

	float t1 = 1.0 + (expf(-1e3 * rDF * s) - 1.0) * v / s;
	vec3 MpR = M + Rayleigh;
	float t3 = rDF * v;
	vec3 t5 = MpR * (AtmosphereDensity / t3);
	vec3 t5x = t5 / expf(x * t3);

	vec3 atmos = (expf(t5 * (t1 - 1.0)) - expf(-t5 + t5x * t1)) / (MpR * t1);
	atmos *= energyFade;
	atmos *= pow(Rayleigh, vec3(5e-3 / (saturate(worldSunVector.y) + 0.009))); 	// Blue tint at twilight

	float rainEnergyLoss = oneMinus(0.65 * wetness);
	//float rainEnergyLoss = oneMinus(0.75 * wetness * wetness);

	atmos *= rainEnergyLoss;

	vec3 color = max0(atmos * (Rayleigh + M * mieAmount * (CornetteShanksPhase(LdotV, 0.9 - wetness * 0.5) * 2.0 + CornetteShanksPhase(LdotV, 0.6 - wetness * 0.4) * 6.0)));

	vec3 ms = oneMinus(expf(t5x - t5)) * rainEnergyLoss;
	color += max0(ms) * expf(SmoothMin(0.0, worldSunVector.y, 0.03) * 2e2) * 0.05;
	
	//color *= oneMinus(wetness * wetness * 0.5) * 7.0;
	color *= oneMinus(wetness * 0.4) * 7.0;

	#ifdef HORIZON_THING
		if (worldViewVector.y < 0.0 && depthFactor > 5.0) {
			color += vec3(0.1, 0.325, 0.5) * expf(-MpR * (floorDist * 0.1 + 0.2 / (max0(worldSunVector.y) + 1e-3))) * oneMinus(wetness/* * wetness*/ * 0.85);
		}
	#endif

	return ColorSaturation(color, 1.0 - wetness * 0.5);
}

// Main sky shading function
vec3 SkyShading(in vec3 worldViewVector) {
	vec3 atmosphere = vec3(0.0);
	if (worldSunVector.y > -0.1) atmosphere += Atmosphere(worldViewVector, worldSunVector, 1.0, AtmosphereExtent);
	if (worldSunVector.y < 0.1) atmosphere += DoNightEye(Atmosphere(worldViewVector, -worldSunVector, 1.0, AtmosphereExtent) * MoonFlux);

	return atmosphere;
}

vec3 SunAbsorptionAtAltitude(in float altitude) {
	float rDFs = AtmosphereDensityFalloff * worldLightVector.y;
	float rDFa = AtmosphereDensityFalloff * altitude;

	return expf(-(Rayleigh + AtmosphereMie) * AtmosphereDensity / (expf(rDFa) * rDFs) - (AtmosphereDensity * expf(-rDFa - 1e3 * rDFs)) / rDFs);
}
/*
vec3 CirrusSunlightColor()
{
	vec3 color = SunAbsorptionAtAltitude(1.5);
	color *= saturate(worldLightVector.y * 40.0);
	if (cloudMoonlit) {
		color *= MoonFlux;
	}
	return color;
}

vec3 CumulusSunlightColor()
{
	vec3 color = SunAbsorptionAtAltitude(1.0);
	color *= saturate(worldLightVector.y * 40.0);
	if (cloudMoonlit) {
		color *= MoonFlux;
	}
	return color;
}
*/
// Lighting colors and data
vec3 GetColorSunlight(in vec3 worldSunVector) {
	vec3 color = vec3(0.0);
	if (worldSunVector.y > -0.1) color += SunlightColorFromSunVector(worldSunVector);
	if (worldSunVector.y < 0.1) color += DoNightEye(SunlightColorFromSunVector(-worldSunVector) * MoonFlux);

	return color;
}

/*
void GetSkylightData(in vec3 worldSunVector,
	out vec4 skySHR, out vec4 skySHG, out vec4 skySHB,
	out vec3 colorSkylight
	//, out vec3 colorSkyUp
)
{
	colorSkylight = vec3(0.0);

	skySHR = vec4(0.0);
	skySHG = vec4(0.0);
	skySHB = vec4(0.0);

	for (int i = 0; i < 5; i++)
	{
		float latitude = float(i) * 0.62831853;
		float cosLatitude = cos(latitude), sinLatitude = sin(latitude);
		for (int j = 0; j < 5; j++)
		{
			float longitude = float(j) * 1.25663706;

			vec3 rayDir;
			rayDir.x = cosLatitude * cos(longitude);
			rayDir.z = cosLatitude * sin(longitude);
			rayDir.y = sinLatitude;

			vec3 skyCol = SkyShading(rayDir, worldSunVector);
			colorSkylight += skyCol;

			skySHR += ToSH(skyCol.r, rayDir);
			skySHG += ToSH(skyCol.g, rayDir);
			skySHB += ToSH(skyCol.b, rayDir);
		}
	}

	skySHR /= 25.0;
	skySHG /= 25.0;
	skySHB /= 25.0;

	colorSkylight /= 25.0;
	//colorSkyUp = SkyShading(vec3(0.0, 1.0, 0.0), worldSunVector);
}
*/

vec3 lightningColor = isLightningFlashing * vec3(0.45, 0.43, 1.0) * 0.03;

#define coneAngleToSolidAngle(x) (TAU * oneMinus(cos(x)))

float fastAcos(float x) {
    float a = abs(x);
	float r = 1.570796 - 0.175394 * a;
	r *= sqrt(1.0 - a);

	return x < 0.0 ? PI - r : r;
}

vec3 physicalSun(in vec3 worldDir, in vec3 sunVector) {
	//const float sunRadius = 1392082.56;
	//const float sunDist = 149.6e6;

	//const float sunAngularRadius = sunRadius / sunDist * 0.5;
	const float sunAngularRadius = TAU / 360.0;

	//const vec3 sunIlluminance = vec3(1.0, 0.973, 0.961) * 126.6e3;
	const vec3 sunIlluminance = vec3(1.026186824, 0.9881671071, 1.015787125) * 5e2;

    float cosTheta = dot(worldDir, sunVector);
    float centerToEdge = 1.0 - max0(sunAngularRadius - fastAcos(cosTheta));
    if (cosTheta < cos(sunAngularRadius)) return vec3(0.0);

	const vec3 alpha = vec3(0.429, 0.522, 0.614); // for AP1 primaries

    vec3 factor = pow(vec3(1.0 - centerToEdge * centerToEdge), alpha * 0.5);
    vec3 finalLuminance = sunIlluminance / coneAngleToSolidAngle(sunAngularRadius) * factor;

	//float visibility = curve(saturate(worldDir.y * 30.0));

    return min(finalLuminance * curve(saturate(worldDir.y * 30.0)), 4e3);
}

vec3 physicalSunReflection(in vec3 worldDir, in vec3 sunVector) {
	const vec3 sunIlluminance = vec3(1.026186824, 0.9881671071, 1.015787125) * 5e2;

    float cosTheta = dot(worldDir, sunVector);
    float centerToEdge = saturate(fastAcos(cosTheta) / 0.05);
    if (cosTheta < cos(0.05)) return vec3(0.0);

	const vec3 alpha = vec3(0.429, 0.522, 0.614); // for AP1 primaries

    float mu = sqrt(1.0 - centerToEdge * centerToEdge);
    vec3 factor = pow(vec3(mu), alpha);
    vec3 finalLuminance = sunIlluminance / coneAngleToSolidAngle(0.05) * factor;

    return min(finalLuminance * curve(saturate(worldDir.y * 30.0)), 5e2);
}

vec3 RenderMoonDiscReflection(in vec3 worldDir, in vec3 sunVector) {
	float cosTheta = dot(worldDir, -sunVector);

	float size = 5e-3;
	float hardness = 2e2;

	float disc = sqr(curve(saturate((cosTheta - 1.0 + size) * hardness)));

	float visibility = curve(saturate(worldDir.y * 10.0));

	return vec3(disc) * visibility * 2.0;
}

vec3 RenderStars(in vec3 worldDir) {
	const float scale = 256.0;
	const float coverage = 0.1 * STARS_COVERAGE;
	const float maxLuminance = 0.6 * STARS_INTENSITY;
	const int minTemperature = 4000;
	const int maxTemperature = 8000;

	//float visibility = curve(saturate(worldDir.y));

	float cosine = worldSunVector.z;
	vec3 axis = cross(worldSunVector, vec3(0.0, 0.0, 1.0));
	float cosecantSquared = rcp(dotSelf(axis));
	worldDir = cosine * worldDir + cross(axis, worldDir) + cosecantSquared * oneMinus(cosine) * dot(axis, worldDir) * axis;

	vec3  p = worldDir * scale;
	ivec3 i = ivec3(floor(p));
	vec3  f = p - i;
	float r = dotSelf(f - 0.5);

	vec3 i3 = fract(i * vec3(443.897, 441.423, 437.195));
	i3 += dot(i3, i3.yzx + 19.19);
	vec2 hash = fract((i3.xx + i3.yz) * i3.zy);
	hash.y = 2.0 * hash.y - 4.0 * hash.y * hash.y + 3.0 * hash.y * hash.y * hash.y;

	float cov = remap(oneMinus(coverage), 1.0, hash.x);
	return maxLuminance * remap(0.25, 0.0, r) * cov * cov * Blackbody(mix(minTemperature, maxTemperature, hash.y));
}

void LandAtmosphericScattering(inout vec3 color, in float dist, in vec3 worldDir) {
	color *= AtmosphereAbsorption(worldDir, dist * 0.005);
	color += Atmosphere(normalize(worldDir), worldSunVector, wetness, dist * mix(0.005, 0.015, wetness))
		* sqr(oneMinus(exp2(-dist * 0.02)));
}
/*
vec2 ProjectSky(in vec3 dir, in float tileSize) {
	float tileSizeDivide = 0.5 * tileSize - 1.5;

	vec2 coord;
	if (abs(dir.x) > abs(dir.y) && abs(dir.x) > abs(dir.z)) {
		dir /= abs(dir.x);
		coord.x = dir.y * tileSizeDivide + tileSize * 0.5;
		coord.y = dir.z * tileSizeDivide + tileSize * (dir.x < 0.0 ? 0.5 : 1.5);
	} else if (abs(dir.y) > abs(dir.x) && abs(dir.y) > abs(dir.z)) {
		dir /= abs(dir.y);
		coord.x = dir.x * tileSizeDivide + tileSize * 1.5;
		coord.y = dir.z * tileSizeDivide + tileSize * (dir.y < 0.0 ? 0.5 : 1.5);
	} else {
		dir /= abs(dir.z);
		coord.x = dir.x * tileSizeDivide + tileSize * 2.5;
		coord.y = dir.y * tileSizeDivide + tileSize * (dir.z < 0.0 ? 0.5 : 1.5);
	}

	return coord * screenPixelSize;
}

vec3 UnprojectSky(in vec2 coord, in float tileSize) {
	//coord *= screenSize;
	float tileSizeDivide = 0.5 * tileSize - 1.5;

	vec3 direction = vec3(0.0);

	if (coord.x < tileSize) {
		direction.x =  coord.y < tileSize ? -1 : 1;
		direction.y = (coord.x - tileSize * 0.5) / tileSizeDivide;
		direction.z = (coord.y - tileSize * (coord.y < tileSize ? 0.5 : 1.5)) / tileSizeDivide;
	} else if (coord.x < 2.0 * tileSize) {
		direction.x = (coord.x - tileSize * 1.5) / tileSizeDivide;
		direction.y =  coord.y < tileSize ? -1 : 1;
		direction.z = (coord.y - tileSize * (coord.y < tileSize ? 0.5 : 1.5)) / tileSizeDivide;
	} else {
		direction.x = (coord.x - tileSize * 2.5) / tileSizeDivide;
		direction.y = (coord.y - tileSize * (coord.y < tileSize ? 0.5 : 1.5)) / tileSizeDivide;
		direction.z =  coord.y < tileSize ? -1 : 1;
	}

	return normalize(direction);
}
*/

vec2 ProjectSky(in vec3 dir) {
	vec2 coord = vec2(atan(-dir.x, -dir.z) * rTAU + 0.5, fastAcos(dir.y) * rPI);
	return saturate(coord * (0.2 - screenPixelSize));
}
vec3 UnprojectSky(in vec2 coord) {
	coord *= rcp(0.2 - screenPixelSize) * vec2(TAU, PI);
	return vec3(sincos(coord.x) * sin(coord.y), cos(coord.y)).xzy;
}
/*
vec2 ProjectSky(vec3 direction) {
	vec2 projectedDir = normalize(direction.xz);

	float azimuthAngle = PI + atan(projectedDir.x, -projectedDir.y);
	float altitudeAngle = PI * 0.5 - fastAcos(direction.y);

	vec2 coord;
	coord.x = azimuthAngle * rTAU;
	coord.y = 0.5 + 0.5 * sign(altitudeAngle) * sqrt(2.0 * rPI * abs(altitudeAngle)); // Section 5.3

	return coord * (0.2 - screenPixelSize);
}

vec3 UnprojectSky(vec2 coord) {
	coord *= rcp(0.2 - screenPixelSize);

	// Non-linear mapping of altitude angle (See section 5.3 of the paper)
	coord.y = (coord.y < 0.5)
		? -sqr(1.0 - 2.0 * coord.y)
		:  sqr(2.0 * coord.y - 1.0);

	float azimuthAngle = coord.x * TAU - PI;
	float altitudeAngle = coord.y * PI * 0.5;

	float altitudeCos = cos(altitudeAngle);
	float altitudeSin = sin(altitudeAngle);
	float azimuthCos = cos(azimuthAngle);
	float azimuthSin = sin(azimuthAngle);

	return vec3(altitudeCos * azimuthSin, altitudeSin, -altitudeCos * azimuthCos);
}
*/
vec2 RaySphereIntersection(in vec3 pos, in vec3 dir, in float rad) {
	float PdotD = dot(pos, dir);
	float delta = sqr(PdotD) + sqr(rad) - dotSelf(pos);

	if (delta < 0.0) return vec2(-1.0);

	delta = sqrt(delta);

	return vec2(-delta, delta) - PdotD;
}
